# 機能設計書 24-コア層（Core Layers）

## 概要

本ドキュメントは、TensorFlow/Kerasにおけるコア層（Core Layers）の機能設計を記述する。Dense・Flatten・Reshape・Lambda・Masking・Dropout・Activation・Permute・RepeatVectorなど、ほぼ全てのニューラルネットワークで使用される基本レイヤーを提供する。

### 本機能の処理概要

**業務上の目的・背景**：ニューラルネットワークの構築には、全結合層（Dense）・形状変換（Reshape, Flatten）・活性化（Activation）・正則化（Dropout）などの基本ビルディングブロックが不可欠である。これらのコア層は、あらゆるモデルアーキテクチャの基盤であり、分類・回帰・生成などすべての深層学習タスクに使用される。

**機能の利用シーン**：画像分類のFC層（Dense）、CNNからRNNへの接続（Flatten/Reshape）、カスタム変換（Lambda）、可変長シーケンスのマスク処理（Masking）、過学習防止（Dropout/SpatialDropout）、入力の次元並べ替え（Permute）、シーケンス繰り返し（RepeatVector）など。

**主要な処理内容**：
1. Dense: 入力に対する全結合線形変換 + 活性化関数適用
2. Flatten: 多次元テンソルを2Dに平坦化
3. Reshape: テンソルの形状を指定形状に変換
4. Lambda: 任意のPython関数をレイヤーとしてラップ
5. Masking: マスク値に基づくタイムステップのマスキング
6. Dropout/SpatialDropout: 学習時のランダムなユニット無効化
7. Activation: 活性化関数の適用

**関連システム・外部連携**：ほぼ全てのKeras機能と連携する。特に結合層（No.23）、活性化関数（No.17）、正則化（No.42）と密接に関連。

**権限による制御**：特になし。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | - | - | 画面機能マッピングに該当なし |

## 機能種別

計算処理（テンソル演算・線形変換・形状変換）

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| units | int | Yes(Dense) | 出力次元数 | 正の整数 |
| activation | str/callable | No | 活性化関数 | 有効な活性化関数名 |
| use_bias | bool | No | バイアスを使用するか | デフォルト: True |
| kernel_initializer | str/callable | No | カーネル初期化方法 | デフォルト: glorot_uniform |
| rate | float | Yes(Dropout) | ドロップアウト率 | 0.0-1.0 |
| target_shape | tuple | Yes(Reshape) | 変換先形状 | バッチ軸除く |
| mask_value | float | No(Masking) | マスク値 | デフォルト: 0.0 |
| function | callable | Yes(Lambda) | ラップする関数 | callableであること |

### 入力データソース

Kerasモデルの前段レイヤー出力。

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| output | Tensor | 各レイヤーの変換結果テンソル |

### 出力先

次段のKerasレイヤーまたはモデル出力。

## 処理フロー

### 処理シーケンス

```
Dense:
1. build: kernel重み[input_dim, units]とbias[units]の作成
2. call: output = activation(dot(input, kernel) + bias)

Flatten:
1. call: channels_firstの場合は転置、その後reshape(-1)

Reshape:
1. call: array_ops.reshape(inputs, target_shape)

Dropout:
1. call: 学習時のみnn.dropout(inputs, rate)適用

Masking:
1. call: mask_value以外の位置のみ出力、マスク情報を_keras_maskに付与
```

### フローチャート

```mermaid
flowchart TD
    A[入力テンソル] --> B{レイヤー種別}
    B -->|Dense| C[matmul + bias + activation]
    B -->|Flatten| D[reshape to 2D]
    B -->|Reshape| E[reshape to target_shape]
    B -->|Dropout| F{training?}
    F -->|Yes| G[nn.dropout適用]
    F -->|No| H[identity]
    B -->|Masking| I[マスク値判定 → マスク生成]
    C --> J[出力]
    D --> J
    E --> J
    G --> J
    H --> J
    I --> J
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-24-01 | Dropout学習時のみ | Dropoutは学習時のみ適用、推論時は無効 | call時のtrainingフラグ |
| BR-24-02 | Lambda変数制限 | Lambda内での変数作成は禁止、警告が出る | Lambda.call実行時 |
| BR-24-03 | Masking伝播 | Maskingのマスクは後続のmasking対応レイヤーに伝播 | supports_masking=True |
| BR-24-04 | Dropoutスケーリング | ドロップされなかったユニットは1/(1-rate)でスケーリング | 学習時 |

### 計算ロジック

- **Dense**: `output = activation(dot(input, kernel) + bias)`
  - kernel: [input_dim, units]
  - bias: [units]（use_bias=Trueの場合）
- **Dropout**: `output = input * mask / (1 - rate)` where mask ~ Bernoulli(1-rate)

## データベース操作仕様

該当なし。

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| ValueError | Dropoutエラー | rateが0-1の範囲外 | 有効な範囲のrateを指定 |
| ValueError | Lambdaエラー | Lambda内で変数を作成 | Layerサブクラスを使用する |
| TypeError | RepeatVectorエラー | nが整数でない | 整数値を指定する |
| ValueError | Reshapeエラー | 変換先形状のサイズが入力と不一致 | target_shapeを修正 |

### リトライ仕様

該当なし。

## トランザクション仕様

該当なし。

## パフォーマンス要件

- Denseは入力がスパース（IndexedSlices）の場合もサポート
- Flattenはeager mode時にconstant_opで高速化
- SpatialDropoutは空間方向に一貫したドロップを実施

## セキュリティ考慮事項

- Lambda層はPythonバイトコードのシリアライズによる保存のため、ポータビリティに制限がある。信頼できないソースからのLambda層の読み込みには注意が必要。

## 備考

- Lambda層よりもLayerサブクラスの使用が推奨される（保存・可視化の観点から）
- SpatialDropout1D/2D/3Dは特徴マップ単位のドロップアウトを実施

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: Masking/Dropout（基本的なデータ前処理層）

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | core.py | `tensorflow/python/keras/layers/core.py` | Masking（69-132行目）: mask_valueの判定とマスク生成 |
| 1-2 | core.py | `tensorflow/python/keras/layers/core.py` | Dropout（135-234行目）: rateの検証、_get_noise_shape、smart_condによる学習/推論切替 |

**読解のコツ**: `control_flow_util.smart_cond`（220-221行目）がKeras内でのtraining flag分岐の標準パターンである。`_keras_mask`プロパティ（123行目）はKeras内部のマスク伝播メカニズムの核心。

#### Step 2: SpatialDropout（拡張ドロップアウト）

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | core.py | `tensorflow/python/keras/layers/core.py` | SpatialDropout1D（237-275行目）: noise_shapeで時間軸を1にする |
| 2-2 | core.py | `tensorflow/python/keras/layers/core.py` | SpatialDropout2D（278-333行目）, SpatialDropout3D（336-390行目） |

#### Step 3: Activation/Reshape/Permute/Flatten（形状変換系）

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | core.py | `tensorflow/python/keras/layers/core.py` | Activation（393-434行目）: activations.get()で活性化関数を取得 |
| 3-2 | core.py | `tensorflow/python/keras/layers/core.py` | Reshape（437-548行目）: _fix_unknown_dimensionでの-1処理 |
| 3-3 | core.py | `tensorflow/python/keras/layers/core.py` | Flatten（608-692行目）: channels_first時の転置処理 |

**主要処理フロー（Flatten）**:
- **644-652行目**: channels_firstの場合、転置でchannels_lastに変換
- **654-658行目**: eager mode時はconstant_opで高速reshape
- **659-675行目**: graph mode時は形状情報を保持したreshape

#### Step 4: RepeatVector/Lambda（特殊変換層）

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 4-1 | core.py | `tensorflow/python/keras/layers/core.py` | RepeatVector（695-737行目）: K.repeat()による繰り返し |
| 4-2 | core.py | `tensorflow/python/keras/layers/core.py` | Lambda（740-998行目）: _check_variablesでの変数使用チェック |

**主要処理フロー（Lambda）**:
- **886-903行目**: call内でGradientTapeとvariable_creator_scopeを使用して変数追跡
- **906-944行目**: 追跡外変数の作成や使用を検出して警告/エラー

### プログラム呼び出し階層図

```
tf.keras.layers.Dense / Flatten / Reshape / ...
    │
    ├─ Layer (base class)
    │      ├─ build(): 重みの作成
    │      ├─ call(): テンソル変換実行
    │      └─ compute_output_shape()
    │
    ├─ Dense.call()
    │      ├─ gen_math_ops.MatMul / sparse_ops.sparse_tensor_dense_matmul
    │      ├─ nn_ops.bias_add()
    │      └─ activations.get()(output)
    │
    ├─ Dropout.call()
    │      └─ smart_cond(training, nn.dropout, identity)
    │
    ├─ Flatten.call()
    │      └─ array_ops.reshape(inputs, [-1, last_dim])
    │
    └─ Lambda.call()
           └─ function(inputs, **kwargs) within GradientTape
```

### データフロー図

```
[入力]                      [処理]                       [出力]

Dense:
[B, input_dim]    ───▶  matmul(x, kernel) + bias  ───▶  [B, units]
                         └─ activation()

Flatten:
[B, H, W, C]     ───▶  reshape                   ───▶  [B, H*W*C]

Dropout:
[B, D]            ───▶  mask * x / (1-rate)       ───▶  [B, D] (学習時)
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| core.py | `tensorflow/python/keras/layers/core.py` | ソース | 全コア層の定義 |
| base_layer.py | `tensorflow/python/keras/engine/base_layer.py` | ソース | Layerベースクラス |
| activations.py | `tensorflow/python/keras/activations.py` | ソース | 活性化関数レジストリ |
| initializers/ | `tensorflow/python/keras/initializers/` | ソース | 重み初期化手法 |
| regularizers.py | `tensorflow/python/keras/regularizers.py` | ソース | L1/L2正則化 |
| constraints.py | `tensorflow/python/keras/constraints.py` | ソース | 重み制約 |
